<?php
/**
 * Create Contact
 *
 * @package     AutomatorWP\Integrations\Groundhogg\Actions\Create_Contact
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Groundhogg_Create_Contact extends AutomatorWP_Integration_Action {

    public $integration = 'groundhogg';
    public $action = 'groundhogg_create_contact';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Create a contact', 'automatorwp-pro' ),
            'select_option'     => __( 'Create a <strong>contact</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Contact. */
            'edit_label'        => sprintf( __( 'Create a %1$s', 'automatorwp-pro' ), '{contact}' ),
            /* translators: %1$s: Contact. */
            'log_label'         => sprintf( __( 'Create a %1$s', 'automatorwp-pro' ), '{contact}' ),
            'options'           => array(
                'contact' => array(
                    'default' => __( 'contact', 'automatorwp-pro' ),
                    'fields' => array(
                        'first_name' => array(
                            'name' => __( 'First Name:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'last_name' => array(
                            'name' => __( 'Last Name:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'email' => array(
                            'name' => __( 'Email:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required' => true,
                        ),
                        'primary_phone' => array(
                            'name' => __( 'Primary Phone:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'primary_phone_extension' => array(
                            'name' => __( 'Phone extension:', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'notes' => array(
                            'name' => __( 'Notes:', 'automatorwp-pro' ),
                            'type' => 'textarea',
                        ),
                        'metas' => array(
                            'name' => __( 'Metas:', 'automatorwp-pro' ),
                            'desc' => __( 'The contact meta values keyed by their meta key.', 'automatorwp-pro' ),
                            'type' => 'group',
                            'classes' => 'automatorwp-fields-table',
                            'options'     => array(
                                'add_button'        => __( 'Add meta', 'automatorwp-pro' ),
                                'remove_button'     => '<span class="dashicons dashicons-no-alt"></span>',
                            ),
                            'fields' => array(
                                'key' => array(
                                    'name' => __( 'Key:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                                'value' => array(
                                    'name' => __( 'Value:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                            ),
                        ),
                    )
                ),
            ),
        ) );

    }

    /**
     * Countries options callback
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function countries_options_cb() {

        $options = array();
        $countries = apply_filters( 'groundhogg-countries', array() );

        foreach( $countries as $country ) {
            $options[$country['code']] = $country['title'];
        }

        return $options;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $contact_data = wp_parse_args( $action_options, array(
            'first_name' => '',
            'last_name' => '',
            'email' => '',
        ) );

        // Bail if email is not correctly setup
        if ( ! is_email( $contact_data['email'] ) ) {
            return;
        }

        // Bail if email exists
        if ( Groundhogg\Plugin::$instance->dbs->get_db( 'contacts' )->exists( $contact_data['email'] ) ) {
            return;
        }

        $contact = new Groundhogg\Contact( $contact_data );

        // Bail if contact not created
        if ( ! $contact->exists() ) {
            return;
        }

        // Update contact meta
        $contact->update_meta( 'primary_phone', $action_options['primary_phone'] );
        $contact->update_meta( 'primary_phone_extension', $action_options['primary_phone_extension'] );
        $contact->add_note( $action_options['notes'] );

        if( is_array( $action_options['metas'] ) ) {

            foreach( $action_options['metas'] as $i => $meta ) {

                // Parse automation tags replacements to both, key and value
                $key = automatorwp_parse_automation_tags( $automation->id, $user_id, $meta['key'] );
                $value = automatorwp_parse_automation_tags( $automation->id, $user_id, $meta['value'] );

                // Sanitize
                $key = sanitize_text_field( $key );
                $value = sanitize_text_field( $value );

                // Update meta meta
                $contact->update_meta( $key, $value );

            }

        }

    }

}

new AutomatorWP_Groundhogg_Create_Contact();